from common import *
import controller
import actions
import time
import math
from pyjoystick.sdl2 import Key, Joystick, run_event_loop  # for buttons

SEQ_NAME = "defaultName"
CURRENT_KEYS = []
COMMAND_HISTORY = []
COMMAND_SAVE = []  # format: ["command", "start", "duration"]
RECORD = False
START_TIME = 0
LAST_TIME = 0
END_TIME = 0


def learn():
    global SEQ_NAME
    if not controller.is_connected():
        log("error", "Controller not connected")
        return
    log("info", "Learning with controller:", controller.getName())
    # ask name
    SEQ_NAME = input("Name of the sequence: ")
    log("info", "TO START: L2")
    log("info", "TO STOP: START + SELECT")
    controller.listen(keyPress)


def keyPress(MAPPING, keyCategory, keyName, keyValue, keyActivation):
    global COMMAND_HISTORY, START_TIME, LAST_TIME, END_TIME, RECORD, CURRENT_KEYS
    if keyName == "L2" and not RECORD and START_TIME == 0:  # START RECORDING
        log("check", "Recording started")
        RECORD = True
    elif set(["START", "SELECT"]).issubset([key[0] for key in CURRENT_KEYS]) and RECORD:  # STOP RECORDING
        log("check", "Recording stopped")
        RECORD = False
        END_TIME = time.time()
        save()
    else:  # SAVE KEYPRESS
        if RECORD:
            if START_TIME == 0:  # first keypress
                START_TIME = time.time()
                LAST_TIME = START_TIME

            # purge old keys values
            for key in CURRENT_KEYS:
                if key[0] == keyName:
                    CURRENT_KEYS.remove(key)

            if keyActivation:  # KEY DOWN
                log("", "Key pressed:", keyName)
                # remove old values (for same key)
                CURRENT_KEYS.append([keyName, keyValue])
            else:  # KEY UP
                log("", "Key released:", keyName)

            # get commands
            commands = actions.GET_COMMANDS(CURRENT_KEYS, False)
            # add commands to history
            for command in commands:
                ALREADY_IN = False
                for c in COMMAND_HISTORY:
                    # if command already in, don't do anything
                    if c['name'] == command['name'] and c['value'] == command['value']:
                        ALREADY_IN = True
                        break
                # if not already in, add it with start time
                if not ALREADY_IN:
                    command['start_time'] = time.time() - START_TIME
                    COMMAND_HISTORY.append(command)

            # check if commands are finished
            for command in COMMAND_HISTORY:
                # if command with same name and value is not in commands, it's finished
                if not any(c['name'] == command['name'] and c['value'] == command['value'] for c in commands):
                    # add command to save
                    command['duration'] = time.time(
                    ) - (command['start_time'] + START_TIME)
                    COMMAND_SAVE.append(command)
                    # remove command from history
                    for c in COMMAND_HISTORY:
                        if c['name'] == command['name'] and c['value'] == command['value']:
                            COMMAND_HISTORY.remove(c)
                            break

            log("check", "COMMAND_HISTORY:", COMMAND_HISTORY)
            log("check", "COMMAND_SAVE:", COMMAND_SAVE)
    if END_TIME == 0:
        # forward normal execution
        controller.keyPress(MAPPING, keyCategory, keyName,
                            keyValue, keyActivation, doLog=False, doSeq=False)


def save():
    global SEQ_NAME, COMMAND_SAVE, START_TIME, END_TIME
    log("info", "Saving", SEQ_NAME, "sequence...")
    content = {
        "name": SEQ_NAME,
        "duration": END_TIME - START_TIME,
        "commands": COMMAND_SAVE
    }
    # save to file
    with open(os.path.join(os.path.dirname(os.path.abspath(__file__)), "sequences/" + SEQ_NAME + ".json"), "w") as file:
        file.write(json.dumps(content, indent=4))

    log("check", "Sequence saved !")
    exit()


def execute(SEQ_NAME):
    global START_TIME
    # open file
    with open(os.path.join(os.path.dirname(os.path.abspath(__file__)), "sequences/" + SEQ_NAME + ".json"), "r") as file:
        content = json.loads(file.read())
        log("", "Reading '" + content['name'] + "' sequence (" +
            str(math.floor(content['duration']))+"s)...")
        # execute commands
        # loop while all commands are not finished
        START_TIME = time.time()
        COMMANDS = content['commands']
        while len(COMMANDS) > 0:
            toExecute = []
            for command in COMMANDS:
                # if never reached add in_execution field
                if not 'in_execution' in command:
                    command['in_execution'] = False
                # check for those to start
                if time.time() - START_TIME >= command['start_time'] and not command['in_execution']:
                    # execute command
                    log("", "Executing command:",
                        command['name'], command['value'])
                    toExecute.append(command)
                    # to avoid executing it again
                    command['in_execution'] = True
                # remove finished commands
                if time.time() - START_TIME >= command['start_time'] + command['duration']:
                    log("check", "Command finished:",
                        command['name'], command['value'])
                    command['value'] = 0
                    toExecute.append(command)
                    COMMANDS.remove(command)
            if len(toExecute) > 0:
                # execute commands
                actions.TRIGGER(toExecute, doLog=False, doSeq=False)


def main():
    global SEQ_NAME
    log("", "Welcome to the lowRider learning mode !")
    log("", "1. Execute a sequence")
    log("", "2. Learn a sequence")
    choice = int(input("Choice: "))
    if choice == 1:
        # list sequences
        log("", "Available sequences:")
        for file in os.listdir(os.path.join(os.path.dirname(os.path.abspath(__file__)), "sequences")):
            if file.endswith(".json"):
                log("", "- " + file[:-5])
                
        SEQ_NAME = input("Name of the sequence: ")
        execute(SEQ_NAME)
    elif choice == 2:
        learn()


if __name__ == "__main__":
    main()
